package com.xith.java3d.overlay;

import javax.media.j3d.*;
import java.awt.image.*;
import java.awt.*;

/**
 * A SubOverlay is one of the pieces which displays a portion of the
 * overlay.  This is used internally by Overlay and should not be referenced
 * directly.
 *
 * Copyright:  Copyright (c) 2000,2001
 * Company:    Teseract Software, LLP
 * @author David Yazel
 *
 */

class SubOverlay {

   int lx,ly,ux,uy;        // which part of the parent image does this represent

   int texWidth;           // the width of the texture (power of 2)
   int texHeight;          // the height of the texture (power of 2)

   int width;              // width of the sub-overlay
   int height;             // height of the sub-overlay

   BufferedImage a;        // buffered image used for icA
   BufferedImage b;        // buffered image used for icB

   ImageComponent2D icA;   // one of the double buffers
   ImageComponent2D icB;   // the other one of the double buffers

   Shape3D shape;          // textured quad used to hold geometry
   Texture2D tex;          // texture mapped to icA or icB
   Appearance ap;          // appearance for this sub-overlay
   Overlay overlay;        // the owner of this sub-overlay

   boolean frontBuffer;    // true if icA is the current texture, false if its icB
   int transferBuffer[];   // used for transferring scan lines from main image to sub-image

   /**
    * Creates the sub overlay for the specified region.
    */
   protected SubOverlay( Overlay overlay, int lx, int ly, int ux, int uy) {

      this.overlay = overlay;

      // save the screen location

      this.lx = lx;
      this.ly = ly;
      this.ux = ux;
      this.uy = uy;

      // caclulate the sub overlay size

      width = ux-lx+1;
      height = uy-ly+1;

      // now we need to calculate the texture size needed to contain this
      // size sub-overlay.  Find the smallest power of two that works

      texWidth = smallestPower(width);
      texHeight = smallestPower(height);

      // create the two buffers

      boolean hasAlpha = (overlay.getBlendAlpha() || overlay.getClipAlpha());
      if (hasAlpha) {
         a = CustomBufferedImage.getCustomRGBA(texWidth,texHeight);
         b = CustomBufferedImage.getCustomRGBA(texWidth,texHeight);
      } else {
         a = CustomBufferedImage.getCustomRGB(texWidth,texHeight);
         b = CustomBufferedImage.getCustomRGB(texWidth,texHeight);
      }

      // determine a compatible by-ref image type

      int icType;
      if (hasAlpha)
         icType = ImageComponent2D.FORMAT_RGBA;
      else
         icType = ImageComponent2D.FORMAT_RGB;

      // create two by-reference image components

      icA = new ImageComponent2D(icType,a,true,true);
      icB = new ImageComponent2D(icType,b,true,true);

      // create a transfer buffer for one scan line

      transferBuffer = new int[texWidth];

      // create the shape

      buildShape();

      // define the texture

      int texType;
      if (hasAlpha) texType = Texture.RGBA;
      else texType = Texture.RGB;

      System.out.println("Sub-overlay : "+width+","+height+" -> "+texWidth+","+texHeight+ " : "+
         lx+","+ly+" - "+ux+","+uy);
      tex = new Texture2D(Texture.BASE_LEVEL,texType,texWidth,texHeight);

      tex.setBoundaryModeS(Texture.CLAMP);
      tex.setBoundaryModeT(Texture.CLAMP);
      tex.setMagFilter(Texture.FASTEST);
      tex.setMinFilter(Texture.FASTEST);
      tex.setImage(0,icA);
      tex.setCapability(Texture.ALLOW_IMAGE_WRITE);
      ap.setTexture(tex);
      frontBuffer = true;


   }

   /**
    * Simple function to return the smallest power of 2 which
    * the value can be contained within
    */

   private int smallestPower( int value ) {

      int n = 1;
      while (n<value)n *=2;
      return n;

   }

   /**
    * Build the quad for this overlay
    */

   private void buildShape() {

      // now we need to calculate the tex coordinates for the upper bounds of
      // the sub-overlay.  This is because we probably had to build the texture
      // bigger than the actual sub-overlay

      float texX = (float)(width) / (float)texWidth;
      float texY = (float)(height) / (float)texHeight;

      float[] verts = {
       ux+1,  ly,  0.0f,
       ux+1,  uy+1,  0.0f,
       lx,  uy+1,  0.0f,
       lx,  ly,  0.0f};

      float[] colors = {
       0.0f, 0.0f, 0.0f, 0.0f,
       0.0f, 0.0f, 0.0f, 0.0f,
       0.0f, 0.0f, 0.0f, 0.0f,
       0.0f, 0.0f, 0.0f, 0.0f};

      float[] tex = {
       texX, 0.0f,
       texX, texY,
       0.0f, texY,
       0.0f, 0.0f, };


      QuadArray planeGeom = new QuadArray(4, QuadArray.COORDINATES |
           QuadArray.TEXTURE_COORDINATE_2);

      planeGeom.setCoordinates(0, verts);
      planeGeom.setTextureCoordinates(0,tex);

      // create the shape and assign the geometry

      shape = new Shape3D();
      shape.setGeometry(planeGeom);

      // assign the appearance

      ap = new Appearance();
      ap.setPolygonAttributes(overlay.getPolygonAttributes());
      ap.setRenderingAttributes(overlay.getRenderingAttributes());
      ap.setTextureAttributes(overlay.getTextureAttributes());
      ap.setTransparencyAttributes(overlay.getTransparencyAttributes());

      Material m = new Material();
      m.setLightingEnable(false);
      ap.setMaterial(m);
      shape.setAppearance(ap);


   }

   /**
    * Transfers the scan lines from the full image to the sub-image. The
    * scan lines are applied in reverse order so that it is compatible with
    * that silly Y-up thing (grrr).
    */
   protected void updateBackBuffer( BufferedImage fullImage ) {

      BufferedImage backBuffer;
      if (frontBuffer) backBuffer = b;
      else backBuffer = a;
      /*
      Graphics g = backBuffer.getGraphics();
      g.setColor(Color.pink);
      g.fillRect(0,0,texWidth-1,texHeight-1);
      g.dispose();
      */


      int w = fullImage.getWidth();
      for (int i=0;i<height;i++) {

         int y = overlay.height-uy+i-1;
         fullImage.getRGB(lx,y,width,1,transferBuffer,0,w);
         backBuffer.setRGB(0,texHeight-i-1-(texHeight-height),width,1,transferBuffer,0,texWidth);

      }
      /*
      Graphics g = backBuffer.getGraphics();
      g.setColor(Color.pink);
      g.drawRect(0,0,width,height);
      g.dispose();
      */

   }

   /**
    * Swaps the buffers
    */

   protected void swap() {
      if (frontBuffer) tex.setImage(0,icB);
      else tex.setImage(0,icA);
      frontBuffer = !frontBuffer;
   }

   /**
    * Return the shape
    */

   public Shape3D getShape() {
      return shape;
   }
}