package com.xith.java3d.overlay;

import javax.media.j3d.*;
import javax.vecmath.*;
import java.awt.*;
import java.awt.image.*;
import java.util.*;

/**
 * A scrolling overlay is built on top of the overlay system.  It maintains
 * optional overlays for the borders of the overlay area and an array of
 * equal size overlays to represent the lines.  This is designed to be fast enough to
 * support fast scrolling text and implementations like a chat box or debug window.
 * A single scroll will effectively swap textures on geometry and stream one overlay's
 * worth of data to the 3d card, so it should be maximally efficient.
 *
 * The design of this implementation is to keep an array of overlays one per virtual line from
 * top to bottom inside the overlay.  Because the position swaps (for scrolling) has to happen
 * within a single behavior to guarentee its transactional status, a single repaint() of the
 * ScrollingOverlay must set the re-order for the lines, even though the actual re-order
 * won't happen until the behavior triggers (on next frame).  We also have to handle the
 * case where several repaints happen between one frame.  In order to facilitate this, we keep
 * an array of double-indirection, mapping overlay-lines to virtual lines, then unwind them
 * when we commit the transaction to the card.
 *
 * So when we first start, or after a commit to the card, you would have a one-to-one mapping
 * of virutal lines to overlays.  If you issue a scroll-up or scroll-down
 * directive, we just adjust the virtual pointers, mapping the first virtual line to the bottom
 * overlay and adjusting from there.  This means you can do multiple scrolls between frames and
 * the buffer maintains consistancy.  If you want a line repaint and a scroll action to be handled
 * atomically, then you should call update() on the scroller.  This will in-turn lock the
 * overlay for update and call the paint() method, which you implement.  This is a bit different
 * than the paint method used at the overlay level since you are not given a graphics context.
 * But from within this method you can call the scroll() and scrollDown() methods and request
 * a graphics context for any virtual line.  You will be writing to the backBuffer of the
 * individual overlay's so it is safe, also none of the line-overlays will commit until you
 * exit the paint() method.
 *
 * Copyright:  Copyright (c) 2000,2001
 * Company:    Teseract Software, LLP
 * @author David Yazel
 *
 */

public class ScrollingOverlay {

   public static final int BORDER_LEFT = 0;
   public static final int BORDER_RIGHT = 1;
   public static final int BORDER_TOP = 2;
   public static final int BORDER_BOTTOM = 3;

   private int width,height;                      // calculated dimensions of the window
   private int numLines;
   private int lineWidth;
   private int lineHeight;
   private Insets margin;

   private Overlay lines[];
   private Point position = new Point();          // desired position
   private boolean visible;                       // desired visibility

   // title overlays

   private Overlay left = null;
   private Overlay right = null;
   private Overlay top = null;
   private Overlay bottom = null;

   // committed information.  If this does not match the desired information on
   // a frame then we will update the scene to match, but within a transaction

   Object mutex = new Object();
   Point curPosition = new Point();    // current position of the window
   boolean curVisible = false;         // true if we are currently visible
   boolean orderChanged;               // true if the line order has changed
   boolean painting;                   // are we painting right now?
   boolean dirty;                      // there has been a change which needs an update
   boolean antialias = false;          // true if the overlay is antialiased

   // the follwing are used to keep track of a line that has been checked out using
   // getLine();

   Overlay curEditLine = null;
   Graphics2D curGraphicsLine = null;

   // java3d nodes needed for ScrollingOverlay

   private BranchGroup consoleBG;            // branch group for overlay
   private Behavior runner;                  // behavior for updating the scrolling overlay

   public ScrollingOverlay(Canvas3D c, TransformGroup viewTG, int lineWidth, int lineHeight,
         int numLines, Insets margin, boolean clipAlpha, boolean blendAlpha) {

      this.numLines = numLines;
      this.lineWidth = lineWidth;
      this.lineHeight = lineHeight;
      this.margin = margin;

      // calulate the dimensions of the window

      width = margin.left + lineWidth + margin.right;
      height = margin.top + lineHeight * numLines + margin.bottom;

      // initialize the overlays for the lines

      lines = new Overlay[numLines];
      for (int i=0;i<numLines;i++) {
         lines[i] = new Overlay(c,viewTG,lineWidth,lineHeight,clipAlpha,blendAlpha,true);
      }

      // create the overlay for the margins

      if (margin.left != 0) left = new Overlay(c,viewTG,margin.left,numLines*lineHeight,clipAlpha,blendAlpha, true);
      if (margin.right != 0) right = new Overlay(c,viewTG,margin.right,numLines*lineHeight,clipAlpha,blendAlpha, true);
      if (margin.top != 0) top = new Overlay(c,viewTG,width, margin.top,clipAlpha,blendAlpha, true);
      if (margin.bottom != 0) bottom = new Overlay(c,viewTG,width, margin.bottom,clipAlpha,blendAlpha, true);


      /**
       * Create a branch group to hold the overlay.  Add the behavior and add all the
       * children overlays
       */
      createBehavior();
      consoleBG = new BranchGroup();
      consoleBG.addChild(runner);

      // add all the children to the branchgroup

      for (int i=0;i<numLines;i++)
         consoleBG.addChild(lines[i].getRoot());

      // add the borders

      if (left != null) consoleBG.addChild(left.getRoot());
      if (right != null) consoleBG.addChild(right.getRoot());
      if (top != null) consoleBG.addChild(top.getRoot());
      if (bottom != null) consoleBG.addChild(bottom.getRoot());

      for (int i=0;i<numLines;i++) {
         lines[i].setBackgroundColor(new Color(0.0f,0.0f,0.0f,0.3f));
      }
      initialize();
      viewTG.addChild(consoleBG);

   }

   /**
    * Implement this to do extra initialization before the node goes live
    */

   public void initialize() {

   }

   public BranchGroup getRoot() {
      return consoleBG;
   }

   /**
    * Returns the number of lines
    */

   public int getNumLines() {
      return numLines;
   }

   public int getLineWidth() {
      return lineWidth;
   }

   public int getLineHeight() {
      return lineHeight;
   }

   public void setAntialias( boolean aa ) {
      antialias = aa;
   }

   /**
    * Sets the visibility of the overlay.  This will be updated in the
    * next frame
    */

   public void setVisible( boolean v ) {
      visible = v;
   }

   /**
    * This method will cause the overlay to be repainted.  This will call
    * the paint method.  Override the paint() method to make changes to the
    * overlay.
    */

   public void repaint() {

      painting = true;
      paint();
      if (curGraphicsLine != null) throw new Error("Call to getLine() did not have a corresponding returnLine()");
      dirty = true;
      painting = false;

   }

   /**
    * Override this method to do any line updates or line re-ordering.  From this method
    * you can call getLine(), scrollUp() or scrollDown().  Your changes will be reflected in
    * the next frame.
    */

   static int count = 0;
   public void paint() {

      // default paint is for testing purposes.

      scrollUp();
      Graphics2D g = getLine(numLines-1);
      g.setColor(Color.black);
//      g.fillRect(0,0,lineWidth,lineHeight);
      g.setColor(Color.white);
      g.drawString("Line "+count++,2,12);
      returnLine();


   }

   /**
    * This will scroll the lines up from the indicated line number.  All the
    * lines above and including the specified line number will be moved up
    * and the topmost line will be rotated down into the gap.  The rotated line
    * will *not* be cleared.  This can only be called from within the paint() method.
    * The lines on the screen will not be updated to reflect the change until the next
    * frame after paint() returns.
    */

   public void scrollUp( int lineNo ) {

      if (!painting) throw new Error("scrollUp() must be called from within paint()");
      if (lineNo==0) return;
      if (lineNo>=numLines) throw new Error("invalid line number specfied");
      Overlay gap = lines[0];
      for (int i=0;i<lineNo;i++)
         lines[i] = lines[i+1];
      lines[lineNo] = gap;
      orderChanged = true;
   }

   public void scrollDown( int lineNo ) {

      if (!painting) throw new Error("scrollDown() must be called from within paint()");
      if (lineNo==numLines-1) return;
      if (lineNo>=numLines) throw new Error("invalid line number specfied");
      Overlay gap = lines[numLines-1];
      for (int i=numLines-1;i>lineNo;i--)
         lines[i] = lines[i-1];
      lines[lineNo] = gap;
      orderChanged = true;
   }

   public void scrollDown() {
      scrollDown(0);
   }
   /**
    * Scrolls all the lines up and rotate the top line to the bottom.
    * The rotated line will *not* be cleared.  This can only be called from
    * within the paint() method. The lines on the screen will not be updated
    * to reflect the change until the next frame after paint() returns.
    */

   public void scrollUp() {
      scrollUp(numLines-1);
   }

   /**
    * This returns a graphics context for painting the specified line.  It is
    * assumed that if you make this call then you will have updated the line.  The
    * backbuffer will be updated with the changes and the overlay will be updated
    * in the next frame.  This can only be called from within the paint() method.
    */

   public Graphics2D getLine( int i ) {

      if (!painting) throw new Error("getLine() must be called from within paint()");
      if (curGraphicsLine != null) throw new Error("Previous call to getLine() or getBorder() did not have a corresponding returnLine()");
      curGraphicsLine = lines[i].getPreppedCanvas();
      if (antialias) {
         curGraphicsLine.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
         curGraphicsLine.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
      }
      curEditLine = lines[i];
      return curGraphicsLine;

   }

   /**
    * This returns a graphics context for painting the specified border.  It is
    * assumed that if you make this call then you will have updated the line.  The
    * backbuffer will be updated with the changes and the overlay will be updated
    * in the next frame.  This can only be called from within the paint() method.
    * You must follow this with a corresponding returnLine() to commit the change.
    * @param border Specfic BORDER_TOP, BORDER_BOTTOM, BORDER_LEFT or BORDER_RIGHT
    */

    public Graphics2D getBorder( int border ) {

      if (!painting) throw new Error("getBorder() must be called from within paint()");
      if (curGraphicsLine != null) throw new Error("Previous call to getLine() or getBorder() did not have a corresponding returnLine()");
      switch (border) {
         case BORDER_LEFT: curEditLine = left;
         case BORDER_RIGHT: curEditLine = right;
         case BORDER_TOP: curEditLine = top;
         case BORDER_BOTTOM: curEditLine = bottom;
      }
      curGraphicsLine = curEditLine.getPreppedCanvas();
      if (antialias) {
         curGraphicsLine.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
         curGraphicsLine.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
      }
      return curGraphicsLine;

    }

   /**
    * This should be called from within the paint() method to update the backbuffer
    * and dispose of the graphics2d.  Failure to return a line after doing a getLine() will
    * result in an exception being thrown when you return from paint()
    */

   public void returnLine() {
      if (!painting) throw new Error("returnLine() must be called from within paint()");
      if (curGraphicsLine == null) throw new Error("returnLine() called without a corresponding getLine()");
      curGraphicsLine.dispose();
      curEditLine.moveToBackbuffer();
      curGraphicsLine = null;
      curEditLine = null;
   }

   /**
    * Sets the position of the window.  It will be updated on the next frame.
    */

   public void setPosition( int x, int y) {
      synchronized(mutex) {
         position.x = x;
         position.y = y;
         dirty = true;
      }
   }

   /**
    * Internally sets the position of all the overlays.  This should only be called if the
    * overlay is invisible or from within a behavior to guarentee that all the sub-pieces
    * are moved together.
    */

   private void setPositionInternal( int x, int y) {

      // save the new current position

      curPosition.x = x;
      curPosition.y = y;

      // adjust the top border

      if (top != null) {
         top.setPosition(x,y);
         y += top.height;
      }

      // adjust the left border

      if (left !=null) {
         left.setPosition(x,y);
      }

      // adjust the right border

      if (right !=null) {
         right.setPosition(x+margin.left+lineWidth,y);
      }

      y+= (numLines*lineHeight);

      // adjust the bottom

      if (bottom !=null) {
         bottom.setPosition(x,y);
      }

      // now sort the lines into order

      orderLines();

   }

   /**
    * Sets all the lines into position according to the array order.
    */
   private void orderLines() {

      int x = curPosition.x + margin.left;
      int y = curPosition.y + margin.top;

      for (int i=0;i<numLines;i++)
         lines[i].setPosition(x,y+(lineHeight*i));
      orderChanged = false;

   }

   /**
    * Sets the visibility of all the overlays.  Should only be called from
    * the behavior.
    */

   private void setVisibleInternal( boolean v ) {

      curVisible = v;
      if (left !=null) left.setVisible(v);
      if (right != null) right.setVisible(v);
      if (top != null) top.setVisible(v);
      if (bottom != null) bottom.setVisible(v);

      for (int i=0;i<numLines;i++) lines[i].setVisible(v);

   }

   /**
    * This should only be called from the behavior.  This updates the overlays,
    * reorders the lines, changes the position and any other currently uncommitted
    * changes
    */

   protected void update() {

      // check the position
      synchronized(mutex) {

         if (!position.equals(curPosition))
            setPositionInternal(position.x, position.y);
         else if (orderChanged) orderLines();

         if (visible != curVisible)
            setVisibleInternal(visible);

         // update all the sub-overlays.  The call to update() is cheap because
         // if there has been no change then there is no work done

         for (int i=0;i<numLines;i++) lines[i].update();
         dirty = false;

      }

   }

   /**
    * Simple behavior to do the buffer swapping.
    */
   private void createBehavior() {

      runner = new Behavior()
      {
         WakeupCriterion wakeupFrames = new WakeupOnElapsedFrames(0,false);
         long time;
         WakeupCriterion wakeup;

         public void initialize()
         {
            wakeup = new WakeupOnElapsedFrames(0,false);
            wakeupOn(wakeup);
         }

         public void processStimulus(java.util.Enumeration enumeration)
         {

            // this should immediatly do an update if it has been
            // at least 300 ms since the last one, but it will not do
            // 2 in a row.

            if (dirty) {
               update();
            }

            wakeupOn(wakeup);
         }
      };
      runner.setSchedulingBounds(new BoundingSphere(new Point3d(0,0,0),100000));


   }

}