package com.xith.java3d.overlay;

import javax.media.j3d.*;
import javax.vecmath.*;
import java.awt.*;
import java.awt.image.*;
import java.util.*;

/**
 * An overlay is 3d geometry which is aligned with the image plate.  This is
 * designed to be as simple as possible for the developer to use.  All that is
 * needed is to extend this class and override the paint function to update the
 * image which will be written displayed on the screen.
 *
 * As a note: Displaying large overlays can use a *huge* amount of texture memory.
 * If you created a pane which was 1024x768 you would consume over 3 mb of texture memory.
 *
 * Another thing to realize is that textures have to be a power of two.  Because of this
 * an overlay which was 513 x 257 would normally have to be generated using 1024 x 512.  Unlike
 * most textures, overlays cannot tolerate stretching and interpolation because of the fuzzyness
 * which would result.  So what the Xith overlay system attempts to do is break the overlay up
 * into small pieces so that extra texture memory is not wasted.  You are given a canvas which
 * represents the entire overlay.  After you finish updating it, the image is ripped apart and
 * written to the underlying textures.  Each of these underlying textures are double-buffered,
 * so that switching them to the java3d engine is quick and does not involve having to continuously
 * allocate and free large BufferedImages.  This means, of course, that for every overlay you
 * use 3 times (width * height * 4bytes) of system memory.  So a chatbox which was 512 x 200
 * would use 3 * ( 512 * 200 *4) bytes or 1.2 MB of system memory.
 *
 * Updates to the actual textures are done within a behavior within a frame.  This is very
 * fast because all that is happening is that the buffers are getting swapped.
 * We have to be certain that we are not in the process of copying the big buffer into the back
 * buffer at the same time the behavior attempts to do a buffer swap. This is handled by the
 * overlay by not updating texture if we are in the middle of drawing a new one.
 * The drawback to this is that numerous updates per second to the overlay could result in
 * several updates not get immediately reflected.  But since the area is always completely
 * redrawn this should not prove to be an issue.  Soon as we hit a frame where we are not updating
 * the buffer then it will be swapped.
 *
 * Remember, all you have to do to make translucent or non-square overlays is
 * to use the alpha channel.
 *
 * Copyright:  Copyright (c) 2000,2001
 * Company:    Teseract Software, LLP
 * @author David Yazel
 *
 */

public class Overlay {

   public static final int BACKGROUND_NONE = 0;
   public static final int BACKGROUND_COPY = 1;


   private static double consoleZ = 2.1f;

   private BufferedImage canvas;             // the large drawing space
   private boolean clipAlpha;                // are we clipping alpha == 0
   private boolean blendAlpha;               // are we blending alpha < 1
   private int imageType;                    // image types for buffers
   private ArrayList subOverlays;            // list of SubOverlay nodes

   protected int width;
   protected int height;

   private double consoleWidth;              // width of console
   private double consoleHeight;             // height of console
   private double scale;                     // scale into world coordinates
   private Dimension canvasDim;              // the dimensions of the Canvas3d
   private Dimension checkDim;               // used to check for dimension changes
   private TransformGroup consoleTG;         // transform group -> screen coords
   private BranchGroup consoleBG;            // branch group for overlay
   private TransformGroup viewTG;            // transform group of the view
   public Transform3D planeOffset;           // transform of image plate
   public Transform3D worldTransform;        // transform to screen space

   // shared resources for the sub-overlays

   private RenderingAttributes ra = null;
   private PolygonAttributes pa = null;
   private TextureAttributes ta = null;
   private TransparencyAttributes tra = null;

   private Behavior runner;

   // the following are used to synchronize the updates to the backbuffer
   // The behavior will do a buffer swap if dirty and not painting.

   boolean dirty = false;
   boolean painting = false;
   boolean owned = false;

   // used for supporting a pre-paint background draw

   int backgroundMode = BACKGROUND_NONE;
   BufferedImage backgroundImage = null;

   // used for locking access to the painting.

   Object mutex;

   // the location of the overlay in screen coordinates

   int posX;
   int posY;
   Canvas3D c3d;

   /**
    * Constructs an overlay window.
    * @param width The width of the window in screen space
    * @param height The height of the window in screen space
    * @param clipAlpha Should we apply a polygon clip where alpha is zero
    * @param blendAlpha Should we blend to background if alpha is < 1
    * @param owned If this overlay is "owned" then no behavior will be built to update it since
    *    it is assumed that the owner will handle the repaints.  Also it will not be
    *    attached to the view transform since it is assumed the owner will do that.
    */
   public Overlay(Canvas3D c, TransformGroup viewTG, int width, int height, boolean clipAlpha,
      boolean blendAlpha, boolean owned) {

      this.owned = owned;
      this.c3d = c;
      this.viewTG = viewTG;
      this.width = width;
      this.height = height;
      this.blendAlpha = blendAlpha;
      this.clipAlpha = clipAlpha;

      canvas = getAppropriateImage();

      // define the rendering attributes used by all sub-overlays

      ra = new RenderingAttributes();
      if (clipAlpha) {
         ra.setAlphaTestFunction(RenderingAttributes.NOT_EQUAL);
         ra.setAlphaTestValue(0);
      }
      ra.setDepthBufferEnable(true);
      ra.setDepthBufferWriteEnable(true);
      ra.setIgnoreVertexColors(true);
      ra.setVisible(false);
      ra.setCapability(RenderingAttributes.ALLOW_VISIBLE_READ);
      ra.setCapability(RenderingAttributes.ALLOW_VISIBLE_WRITE);

      // define the polygon attributes for all the sub-overlays

      pa = new PolygonAttributes();
      pa.setBackFaceNormalFlip(false);
      pa.setCullFace(PolygonAttributes.CULL_NONE);
      pa.setPolygonMode(PolygonAttributes.POLYGON_FILL);

      // define the texture attributes for all the sub-overlays

      ta = new TextureAttributes();
      ta.setTextureMode(TextureAttributes.REPLACE);
      ta.setPerspectiveCorrectionMode(TextureAttributes.FASTEST);

      // if this needs to support transparancy set up the blend

      if (clipAlpha || blendAlpha) {
         tra = new TransparencyAttributes(TransparencyAttributes.BLENDED,1.0f);
         ta.setTextureBlendColor(new Color4f(0,0,0,1));
      }

      // define the branch group where we are putting all the sub-overlays

      consoleBG = new BranchGroup();
      consoleTG = new TransformGroup();
      consoleTG.setCapability(consoleTG.ALLOW_TRANSFORM_WRITE);
      consoleBG.addChild(consoleTG);
      consoleTG.setTransform(new Transform3D());

      canvasDim = new Dimension();
      checkDim = new Dimension();
      planeOffset = new Transform3D();

      // set the position defaulted to 0,0

      setPosition(0,0);

      // now we need to calculate the sub-overlays needed for the overlay

      System.out.println("Overlay : "+width+"x"+height);
      createSubOverlays();
      if (!owned) createBehavior();

      // attach the behavior to the branchgroup

      consoleTG.addChild(runner);
      initialize();
      if (!owned) viewTG.addChild(consoleBG);
      mutex = new Object();


   }

   public Overlay(Canvas3D c, TransformGroup viewTG, int width, int height, boolean clipAlpha, boolean blendAlpha) {
      this(c,viewTG,width,height,clipAlpha,blendAlpha,false);
   }

   /**
    * This should be overwritten if you want to add any additional objects or
    * behaviors or initialization before the overlay goes live
    */
   protected void initialize() {

   }

   /**
    * This sets the position of the overlay in screen coordinates. 0,0 is the
    * upper left of the screen
    */

   public void setPosition( int x, int y) {

//      System.out.println("Overlay position : "+x+"x"+y);
      posX = x;
      posY = y;

      // get the field of view and then calculate the width in meters of the
      // screen

      double fov = c3d.getView().getFieldOfView();
      consoleWidth = ((double)java.lang.Math.tan (fov / 2.0) * consoleZ)*2.0 ;

      // get the canvas width in pixels, then calculate the scale from
      // pixels to meters.  Then calculate the height of the screen in meters

      c3d.getSize(canvasDim);
//      canvasDim.width= 1024;
//      canvasDim.height = 768;
      scale = consoleWidth / (double)canvasDim.width;
      consoleHeight = canvasDim.height * scale;

      // now we need to calculate the adjustment for the position of the overlay
      // relative to the screen.  Also reverse the Y coordinate so that 0,0 is
      // upper left

      double xDelta = scale * (double)x;
//      double yDelta = scale * (double)y;
      double yDelta = scale * (double)(canvasDim.height-y-height);

      // build the plane offset

//      System.out.println("Overlay console : "+consoleWidth+"x"+consoleHeight);
//      System.out.println("Overlay scale : "+scale);
      planeOffset.setTranslation(new Vector3d(-consoleWidth/2+xDelta, -consoleHeight/2+yDelta, (double)-consoleZ));
      planeOffset.setScale(scale);
      consoleTG.setTransform(planeOffset);

   }

   /**
    * Return the root of the overlay and its sub-overlays so it can be
    * added to the scenegraph
    */

   public BranchGroup getRoot() {
      return consoleBG;
   }

   /**
    * This internal method creates the sub-overlays for the overlay.  It will
    * attempt to create an optimal arrangement of textures.
    */

   private void createSubOverlays() {

      subOverlays = new ArrayList();

      // calculate the number of columns and their sizes

      int numCols = 0;
      int cols[] = new int[5];
      int w = width;
      while (w>0) {

         int p = optimalPower(w,32)-1;
         if (p>w) p = w;
         cols[numCols] = p;
         w -= cols[numCols++];
      }

      // now calculate the number of rows and their sizes

      int numRows = 0;
      int rows[] = new int[5];
      int h = height;
      while (h>0) {

         int p = optimalPower(h,32);
         if (p>h) p = h;
         rows[numRows] = p;
         h -= rows[numRows++];
      }

      // now we have the optimal grid we need to create the sub-overlays.
      // The bounds passed to the sub-overlay are 0,0 based since they represent
      // pieces of the big canvas

      /*
      int ly = 0;
      for (int row=0;row<numRows;row++) {

         int lx=0;
         int xStart = 0;
         int xStop = cols[0]-1;
         for (int col=0;col<numCols;col++) {

            int ya = (height-(ly+rows[row])-1);
            int yb = height-ly-1;
            SubOverlay s = new SubOverlay(this,xStart,ya,xStop,yb);
            subOverlays.add(s);
            consoleTG.addChild(s.getShape());

            if (col<numCols-1) {
               xStart = xStop;
               xStop = xStop+cols[col+1];
            }

         }
         ly += rows[row]-1;
      }
      */

      int yStart = 0;
      int yStop = rows[0]-1;
      int ly = 0;
      for (int row=0;row<numRows;row++) {

         int xStart = 0;
         int xStop = cols[0]-1;
         for (int col=0;col<numCols;col++) {

            SubOverlay s = new SubOverlay(this,xStart,yStart,xStop,yStop);
            subOverlays.add(s);
            consoleTG.addChild(s.getShape());

            if (col<numCols-1) {
               xStart = xStop+1;
               xStop = xStop+cols[col+1];
            }

         }
         if (row<numRows-1) {
            yStart = yStop+1;
            yStop = yStop+rows[row+1];
         }
      }
   }

   /**
    * Returns an optimal power of two for the value given.  It will either
    * return the largest power of 2 which is less than or equal to the value, OR
    * it will return a larger power of two as long as the difference between
    * that and the value is not greater than the threshhold.
    */

   private int optimalPower( int value, int threshhold ) {

      int n = 1;
      while (n<value) {

         if (n*2>256) return 256;
         if (n*2 > value) {

            if (n*2-value < threshhold) return n*2;
            else return n;
         }
         n *= 2;
      }
      return n;

   }

   /**
    * Synchronize with the view platform
    */
   public void setWorldTransform() {

      c3d.getSize(checkDim);
      if (!checkDim.equals(canvasDim)) {
         setPosition(posX,posY);
      }

   }

   /**
    * Returns true if the overlay is clipping when alpha is zero
    */

   public boolean getClipAlpha() {
      return clipAlpha;
   }

   /**
    * Returns true if the overlay is blending alpha.
    */

   public boolean getBlendAlpha() {
      return blendAlpha;
   }

   /**
    * Returns the image type used for the buffered image
    */

   public int getImageType() {
      return imageType;
   }

   /**
    * Return the rendering attributes shared by all sub-overlays
    */

   public RenderingAttributes getRenderingAttributes() {
      return ra;
   }

   /**
    * Return the polygon attributes shared by all the sub-overlays
    */

   public PolygonAttributes getPolygonAttributes() {
      return pa;
   }

   /**
    * Return the texture attributes shared by all the sub-overlays
    */

   public TextureAttributes getTextureAttributes() {
      return ta;
   }

   /**
    * Return the transparency attributes
    */

   public TransparencyAttributes getTransparencyAttributes() {
      return tra;
   }

   /**
    * Move the contents of the drawing canvas into the various
    * backbuffers.
    */
   protected void moveToBackbuffer() {

      synchronized (mutex) {

         // copy all the buffers to the back buffer

         painting = true;
         int n = subOverlays.size();
         for (int i=0;i<n;i++) {
            SubOverlay s = (SubOverlay)subOverlays.get(i);
            s.updateBackBuffer(canvas);
         }

         // flag the frame as dirty

         dirty = true;
         painting = false;
      }

   }

   /**
    * Prepares the canvas to be painted.  This should only be called internally
    * or from an owner like the ScrollingOverlay class
    */

   protected Graphics2D getPreppedCanvas() {

      if (backgroundMode == BACKGROUND_COPY) {

         if (backgroundImage != null) {
            canvas.setData(backgroundImage.getRaster());
         }
      }

      Graphics2D g = (Graphics2D)canvas.getGraphics();
      return g;

   }

   /**
    * This is called to trigger a repaint of the overlay.  This will return once
    * the back buffer has been built, but before the swap.
    */

   public void repaint() {

      Graphics2D g = getPreppedCanvas();
      paint(g);
      g.dispose();

      moveToBackbuffer();

   }

   /**
    * This method is ONLY called from the behavior.  This checks to see if it needs
    * to swap the buffers.  If so, it locks the mutex and swaps all the sub-overlay
    * buffers at once.  If this isn't called from a behavior then the sub-buffer swaps
    * might happen in seperate frames.
    */

   protected void update() {


      if (dirty && !painting) {

         synchronized (mutex) {

            // swap all the sub-buffers

            int n = subOverlays.size();
            for (int i=0;i<n;i++) {
               SubOverlay s = (SubOverlay)subOverlays.get(i);
               s.swap();
            }
            dirty = false;
         }

      }
   }

   /**
    * Changes the visibility of the overlay.
    */

   public void setVisible( boolean yes ) {
      ra.setVisible(yes);
   }
   /**
    * This is where the actualy drawing of the window takes place.  Override
    * this to alter the contents of what is show in the window.
    */

   public void paint(Graphics2D g) {

      // set up a test pattern for checking for distortions

      g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
      /*
      g.setColor(Color.black);
      g.fillRect(0,0,width,height);
      */
      g.setColor(Color.red);
      g.fillOval(1,1,width-2,height-2);

      g.setColor(Color.yellow);
      g.drawLine(0,height-1,width-1,height-1);

      g.setColor(Color.cyan);
      g.drawLine(0,0,width-1,0);

      g.setColor(Color.green);
      g.drawLine(0,0,0,height-1);

      g.setColor(Color.magenta);
      g.drawLine(width-1,0,width-1,height-1);

   }

   /**
    * Protected method to get the view which this overlay is matching
    */

   protected View getView() {
      return c3d.getView();
   }
   /**
    * Allows an derived class to add a behavior or object to the same transform
    * group that the sub-overlays use
    */

   protected void attachNode( Node node ) {
      consoleTG.addChild(node);
   }

   /**
    * Simple behavior to do the buffer swapping.
    */
   private void createBehavior() {

      runner = new Behavior()
      {
         WakeupCriterion wakeupFrames = new WakeupOnElapsedFrames(0,false);
         WakeupCriterion wakeup;

         public void initialize()
         {
            wakeup = new WakeupOnElapsedFrames(0,false);
            wakeupOn(wakeup);
         }

         public void processStimulus(java.util.Enumeration enumeration)
         {

            // this should immediatly do an update if it has been
            // at least 300 ms since the last one, but it will not do
            // 2 in a row.

            if (dirty) {
               update();
            }

            setWorldTransform();
            wakeupOn(wakeup);
         }
      };
      runner.setSchedulingBounds(new BoundingSphere(new Point3d(0,0,0),100000));


   }

   /**
    * Internal routine to return an appropriate buffered image for the overlay
    * size and format.
    */
   private BufferedImage getAppropriateImage() {

       if (clipAlpha || blendAlpha)
         return new BufferedImage(width,height,BufferedImage.TYPE_INT_ARGB);
      else
         return CustomBufferedImage.getCustomRGB(width,height);

   }

   /**
    * Returns a BufferedImage appropriate for using as a background.  It will be the same
    * size as the overlay with an appropriate format.  If one has already been assigned to
    * this overlay then that is what will be returned, otherwise one is built and returned.
    * If a new one is built it is NOT assigned as a background image. To do that call
    * setBackgroundImage()
    */

   public BufferedImage getBackgroundImage() {
      if (backgroundImage != null) return backgroundImage;
      else return getAppropriateImage();
   }

   /**
    * Sets the background to a solid color.  This will be set before every call to
    * paint() to set the drawing surface.  This actually builds a background image matching
    * the size and format of the overlay and then initializes it to the color.  If a background
    * image already exists then it will be initialized to to the color.  It is completely
    * appropriate to have an alpha component in the color if this is a alpha
    * capable overlay.  In general you should only use background images if this is an
    * overlay that is called frequently, since you could always paint it inside your
    * paint() method.  This is also designed to support the same background image
    * used for multiple overlays, like in a scrolling overlay.
    */

   public void setBackgroundColor( Color c ) {

      if (backgroundImage == null) backgroundImage = getAppropriateImage();
      int pixels[] = new int[width*height];
      int rgb = c.getRGB();
      for (int i=0;i<width*height;i++)
         pixels[i] = rgb;
      backgroundImage.setRGB(0,0,width,height,pixels,0,width);
      backgroundMode = BACKGROUND_COPY;
   }


   /**
    * Sets the background image to the one specified.  It does not have to be the same size as the
    * overlay but the it should be at least big enough is the backgroundMode is BACKGROUND_COPY,
    * since that is a straight raster copy.  Setting this to null will remove the background
    * image.
    */

   public void setBackgroundImage(BufferedImage bi) {
      backgroundImage = bi;
      if (bi==null) backgroundMode = BACKGROUND_NONE;
      else backgroundMode = BACKGROUND_COPY;
   }

   /**
    * Sets the background mode.  BACKGROUND_COPY will copy the raster data from the background
    * into the canvas before paint() is called.  BACKGROUND_NONE will cause the background
    * to be disabled and not used.
    */

   public void setBackgroundMode( int mode ) {
      backgroundMode = mode;
   }
}