package com.xith.client.gui;

import com.xith.java3d.overlay.*;
import javax.media.j3d.*;
import java.awt.*;
import java.awt.image.*;
import java.util.*;
import java.awt.event.*;
import java.text.*;
import java.awt.font.*;
import com.navtools.util.WindowsTimeProvider;
import javax.vecmath.*;

/**
 * Chat overlay box.  Highly optimized for 3d viewing this chat box uses a unique
 * system to synchronize changes to the chat log and text entry.
 *
 * Design:
 *
 * Uses a ScrollingOverlay to manage a window with fast scrolling lines.  Scrolling
 * lines are at almost zero cost and the update of one line can happend at single
 * frame speeds.  A queue of messages are kept which hold the instructions to the chat
 * box.  The paint() method steps through the queue and applies a bunch of updates to the
 * chat box and the next frame will reflect the changes to the screen.
 *
 * Copyright:  Copyright (c) 2000,2001
 * Company:    Teseract Software, LLP
 * @author David Yazel
 *
 */

public class ChatOverlay extends ScrollingOverlay {

   private static final int CHAT_KEY = 0;          // add a key to text entry
   private static final int CHAT_LOG = 1;          // add a string to the log
   private static final int CHAT_BORDER = 2;       // redraw the border
   private static final int CHAT_BACKSPACE = 3;    // backspace in chat entry
   private static final int CHAT_REDRAW = 4;       // redraw everything including border
   private static final int CHAT_PAGEUP = 5;
   private static final int CHAT_PAGEDOWN = 6;
   private static final int CHAT_ENTERCHAT = 7;
   private static final int CHAT_EXITCHAT = 8;
   private static final int CHAT_INIT = 9;

   // different font styles

   public static final int FONT_PLAIN = 0;
   public static final int FONT_ITALICS = 1;
   public static final int FONT_BOLD = 2;

   private Vector msgQ;                         // list of ChatMessage
   private float transparency;                  // how transparent is background
   private ArrayList chatHistory;               // list of TextNode
   boolean chatMode = false;                    // whether we are in chat mode
   String entryLine = "";                       // the current line being entered
   private Behavior runner;                  // behavior for updating the scrolling overlay

   ChatFont fonts[];             // array of legitimate fonts
   int wrapIndent = 15;          // how far to indent after the first line

   /**
    * Colors used for drawing the chatbox
    */
   Color bgColor = Color.black;
   Color bgBorderColor = Color.darkGray;
   Color bgEntryColor = new Color(0.15f,0.15f,0.15f);
   Color fgEntryColor = Color.lightGray;
   Color bgEntryBorderColor = Color.gray;

   public ChatOverlay(Canvas3D c, TransformGroup viewTG, int lineWidth, int numLines) {
      super(c,viewTG,lineWidth,16,numLines,new Insets(3,3,3,3),false,true);
   }

   public void initialize() {

      // create the default fonts

      fonts = new ChatFont[3];
      fonts[FONT_PLAIN] = new ChatFont("Helvetica",12,Font.PLAIN);
      fonts[FONT_ITALICS] = new ChatFont("Helvetica",12,Font.ITALIC);
      fonts[FONT_BOLD] = new ChatFont("Helvetica",12,Font.BOLD);

      // create the message queue.  Using sychronized object to protect
      // the queue from other threads.

      msgQ = new Vector();
      chatHistory = new ArrayList();

      // define a message for drawing the borders, which generally only needs
      // to be done initially

      ChatMessage m = new ChatMessage(CHAT_INIT);
      msgQ.add(m);
      createBehavior();

      getRoot().addChild(runner);

   }

   /**
    * Send some simple commands into the queue to simulate chatting
    */
   public void runTest() {

      postText(Color.green,"Testing first line",FONT_PLAIN);
      postText(Color.yellow,"Testing second line, should be bold",FONT_BOLD);
      postText(Color.cyan,"Testing some very long text to see if word wrapping is acting as it should be.  This should create a couple of lines.  And so forth and so on, from time to time",FONT_PLAIN);
      postText(Color.green,"Testing another line line",FONT_PLAIN);
      postText(Color.green,"Anyone up for a battle?",FONT_ITALICS);
      postText(Color.green,"Move UP! Incomming!",FONT_PLAIN);
      postText(Color.green,"Yet another line!",FONT_PLAIN);
      enableChatMode();
      postKeystrokes("Listen my children and you shall hear of the midnight ride of Paul revere");
      postText(Color.green,"Ack they got me, fall back!",FONT_PLAIN);
      backspace();
      backspace();
      backspace();
      backspace();
      backspace();
      backspace();
      backspace();
      backspace();
      backspace();
      disableChatMode();

   }

   /**
    * This can draw one line of text.  This assumes that it fits on the line
    * and has already been word wrapped and broken up into lines.
    */

   private void drawLine(Graphics2D g, TextNode node ) {

      g.setColor(node.color);
      ChatFont f = fonts[node.mode];

      AttributedString as = new AttributedString(node.text);
      as.addAttribute(TextAttribute.FONT, f.getFont(g), 0, node.text.length());
      AttributedCharacterIterator si = as.getIterator();
      TextLayout tl = new TextLayout(si,g.getFontRenderContext());

      tl.draw(g,0,f.getHeight(g)-f.getDescent(g));

   }


   /**
    * Add a line to the log.  Generally this just means scrolling up and
    * writing the new line, but if the line is too long then it needs to be
    * broken up into pieces.  Also, if we are in chat mode then we need to scroll
    * up from the 2nd to last line rather than the last line
    */

   private void addToChatLog( ChatMessage m ) {

      int line = getNumLines()-1;
      if (chatMode) line--;

      // write out as many lines as necessary to print the text,
      // wrapping as we go.  Use the attributed character iterator
      // so we can wrap by words, and so later we can add types of
      // langauges and markup

      AttributedString as = new AttributedString(m.text.text);
      AttributedCharacterIterator si = null;

      LineBreakMeasurer lb = null;
      float wrappingWidth = getLineWidth() -10;
      float x = 0;
      int pos = 0;
      do {

         // scrollup and get the graphics for this line

         scrollUp(line);
         Graphics2D g = getLine(line);

         // get the line break measurer if this is the first one

         ChatFont cf = fonts[m.text.mode];
         Font f = cf.getFont(g);
         g.setFont(f);
         if (lb==null) {
            as.addAttribute(TextAttribute.FONT, f, 0, m.text.text.length());
            si = as.getIterator();
            lb = new LineBreakMeasurer(si,g.getFontRenderContext());
         }

         g.setColor(m.text.color);
         TextLayout layout = lb.nextLayout(wrappingWidth);
         int nextPos = lb.getPosition();
         layout.draw(g,x,cf.getHeight(g)-cf.getDescent(g));
         returnLine();

         // now put the sub-line into chat history

         String sub = m.text.text.substring(pos,nextPos);
         TextNode historyNode = new TextNode(sub,m.text.color,m.text.mode);
         chatHistory.add(historyNode);

         // now set the wrap width to be smaller for the indent

         if (x==0) {
            x = wrapIndent;
            wrappingWidth -= wrapIndent;
         }

      } while (lb.getPosition()<m.text.text.length());


   }

   /**
    * Draws the borders and blank lines
    */

   private void initChatBox() {

      for (int i=0;i<getNumLines();i++) {

         getLine(i);
         returnLine();

      }

   }

   /**
    * The entry line is drawn on the bottom line with a box around it.  If the
    * text is too long to fit then it is scrolled to the left.
    */

   private void drawEntryLine() {

      Graphics2D g = getLine(getNumLines()-1);

      int w = getLineWidth();
      int h = getLineHeight();

      g.setColor(bgEntryBorderColor);
      g.drawRect(0,0,w-1,h-1);

      if (entryLine.length()>0) {

         g.setColor(fgEntryColor);
         ChatFont f = fonts[FONT_PLAIN];
         g.setFont(f.getFont(g));

         AttributedString as = new AttributedString(entryLine);
         as.addAttribute(TextAttribute.FONT, f.getFont(g), 0, entryLine.length());
         AttributedCharacterIterator si = as.getIterator();
         TextLayout tl = new TextLayout(si,g.getFontRenderContext());

         tl.draw(g,1,f.getHeight(g)-f.getDescent(g));
      }

      returnLine();

   }

   /**
    * Removes the chat box from entry mode and scrolls everything back down.
    */

   private void exitChat() {
      chatMode = false;
      scrollDown();
      Graphics2D g = getLine(0);

      int i = chatHistory.size()-getNumLines();
      if (i>=0) {
         TextNode node = (TextNode)chatHistory.get(i);
         drawLine(g,node);
      }
      returnLine();
   }

   /**
    * Entering chat scrolls up the box, sets the chatMode flag and draws the blank
    * entry screen.
    */

   private void enterChat() {

      chatMode = true;
      scrollUp();
      entryLine = "";
      drawEntryLine();

   }

   /**
    * Use the chatManager to paint the chat overlay area
    */
   public void paint() {

      // consume a single message on the message queue and process it.  There
      // could certainly be more ready to go, but most messages involve the
      // update of at least one line and we should really try to not update
      // more than one line per frame.  This is because for each line update
      // we are sending a texture over to the 3d card.

      if (msgQ.size() >0 ) {

         ChatMessage m = (ChatMessage)msgQ.remove(0);
         switch (m.msgType) {

            case CHAT_LOG: addToChatLog(m); break;
            case CHAT_INIT: initChatBox(); break;
            case CHAT_ENTERCHAT: enterChat(); break;
            case CHAT_EXITCHAT: exitChat(); break;
            case CHAT_BACKSPACE:
               if (entryLine.length()>0) {
                  entryLine = entryLine.substring(0,entryLine.length()-1);
                  drawEntryLine();
               }
               break;

            case CHAT_KEY:
               entryLine += m.key;
               drawEntryLine();
               break;

         }

      }

   }

   public void postKeystrokes(String text) {

      for (int i=0;i<text.length();i++) {
         msgQ.add(new ChatMessage(CHAT_KEY,text.charAt(i)));
      }


   }

   /**
    * This will post a line to the chat history.  This will wrap lines that are too long
    * to fit on a single line.
    */

   public void postText( Color color, String text, int style) {

      ChatMessage m = new ChatMessage(CHAT_LOG);
      m.text = new TextNode(text,color,style);
      msgQ.add(m);

   }

   /**
    * Removes a single char from the end of the entry line.  If we are at the
    * beginning of the line nothing will happen.
    */

   public void backspace() {
      msgQ.add(new ChatMessage(CHAT_BACKSPACE));
   }

   /**
    * Adds the specified keypress to the end of the entry line.
    */
   public void keypress( KeyEvent key ) {
      ChatMessage m = new ChatMessage(CHAT_KEY);
      m.key = key.getKeyChar();
      msgQ.add(m);
   }

   public void enableChatMode() {
      msgQ.add(new ChatMessage(CHAT_ENTERCHAT));
   }

   public void disableChatMode() {
      msgQ.add(new ChatMessage(CHAT_EXITCHAT));
   }

   /**
    * Returns the text being entered.
    */

   public String getEntryLine() {
      return entryLine;
   }

   /**
    * Sets the entry line to the specified text.  This will only show up
    * if chatMode is true.
    */

   public void setEntryLine(String text) {
      this.entryLine = text;
   }

   /**
    * Holds information on a single line of text.  The line of text should not
    * be longer than what fits in the text window and should only be one line.
    * Wrapping is handled by breaking the longer line when the line is inserted into
    * the chat history.
    */

   class TextNode {

      public String text;     // text to be printed
      public Color color;     // color to use
      public int mode;        // text mode : TEXT_NORMAL, TEXT_ITALICS or TEXT_BOLD

      public TextNode(  String text, Color color, int mode ) {
         this.color = color;
         this.text = text;
         this.mode = mode;
      }

   }

   /**
    * A message class used to store a single message needed by the chat window.
    */

   class ChatMessage {

      public int msgType;           // message types;
      public char key;              // for single key press CHAT_KEY
      public TextNode text = null;  // for adding text into the log

      public ChatMessage(int mess) {
         msgType = mess;
      }

      public ChatMessage(int mess, char ch) {
         msgType = mess;
         key = ch;
      }

   }

   /**
    * A node used to store information on a font being used by the chatbox.  The
    * chatbox uses 3 or more fonts.  In particular we want to pre-calculate the
    * font metrics for the display context, which we cannot do until we
    * are in paint.
    */

   class ChatFont {
      Font font;              // actual font
      int height;            // total height of the font
      int descent;           // descent for the font
      boolean changed;        // if the font has changed we need to recalc
      String fontName;        // what font is this
      int size;               // what is the point size
      int style;              // ITALIC, BOLD, PLAIN

      ChatFont( String fontName, int size, int style ) {
         this.fontName = fontName;
         this.size = size;
         this.style = style;
         this.changed = true;
      }

      /**
       * Takes the font information and builds the font, then
       * calculates the font metrics.
       */
      private void getFontMetrics(Graphics g) {

         font = new Font(fontName,style,size);
         g.setFont(font);
         height = g.getFontMetrics().getHeight();
         descent = g.getFontMetrics().getMaxDescent();
         changed = false;
      }

      public Font getFont(Graphics g) {
         if (changed) getFontMetrics(g);
         return font;
      }

      public int getDescent( Graphics g ) {
         if (changed) getFontMetrics(g);
         return descent;
      }

      public int getHeight(Graphics g) {
         if (changed) getFontMetrics(g);
         return height;
      }
   }

   /**
    * Simple behavior to do the buffer swapping.
    */
   private void createBehavior() {

      runner = new Behavior()
      {
         WakeupCriterion wakeupFrames = new WakeupOnElapsedFrames(0,false);
         WindowsTimeProvider timer = new WindowsTimeProvider();
         long time;
         WakeupCriterion wakeup;

         public void initialize()
         {
            wakeup = new WakeupOnElapsedFrames(0,false);
            time = timer.currentTimeMillis();
            wakeupOn(wakeup);
         }

         public void processStimulus(java.util.Enumeration enumeration)
         {

            // this should immediatly do an update if it has been
            // at least 300 ms since the last one, but it will not do
            // 2 in a row.

            long newTime = timer.currentTimeMillis();
            if ((newTime-time>50) && (msgQ.size()>0)) {
               repaint();
               time = newTime;
            }

            wakeupOn(wakeup);
         }
      };
      runner.setSchedulingBounds(new BoundingSphere(new Point3d(0,0,0),100000));


   }

}