import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.awt.image.*;
import java.beans.*;

public class PanelAgent extends Agent
  implements SelectionListener, ActionListener
{
	private JFrame frame;
	private JMenu menu;
	private SelectionMaker selectMaker = new SelectionMaker();
	private JPanel agentPanel;
	private Agent master;

	public PanelAgent(final Agent master)
	{
		super();
		this.master = master;
		selectMaker.addSelectionListener(this);

		agentPanel = new JPanel()
		 {
			public Color backgroundColor = Color.white;

			public void paint(Graphics g)
			{
				if(g != null)
				{
					g.setColor(backgroundColor);
					g.fillRect(0, 0, g.getClipBounds().width, g.getClipBounds().height);
					master.paint(g);
					selectMaker.paint(g);
				}
			}
		 };

		agentPanel.addComponentListener(new ComponentAdapter()
		 {
			public void componentResized(ComponentEvent e)
			{
				Signal s = new Signal(Signal.SHAPE_CHANGED);
				s.content = agentPanel.getSize();
				master.signal(s);
			}
		 });

		agentPanel.addMouseMotionListener(new MouseMotionAdapter()
		 {
			public void mouseDragged(MouseEvent e) { step(); }
		 });

		agentPanel.addMouseListener(new MouseAdapter()
		 {
			public void mouseReleased(MouseEvent e) { step(); }
		 });

		agentPanel.addMouseListener(selectMaker);
		agentPanel.addMouseMotionListener(selectMaker);
	}

	public JPanel getPanel()
	{
		return agentPanel;
	}

	public JFrame getFrame()
	{
		if(frame == null)
		{
			frame = new JFrame();
			frame.setJMenuBar(getMenuBar());
			frame.getContentPane().add(agentPanel);
			frame.setTitle("Agent Drawing Test");
			frame.addWindowListener(new WindowAdapter()
			 {
				public void windowClosing(WindowEvent e)
				{
					e.getWindow().setVisible(false);
					e.getWindow().dispose();
					System.exit(0);
				}
			 });
			frame.pack();
		}
		return frame;
	}

	public JMenuBar getMenuBar()
	{
		if(menu == null)
			createMenu();
		JMenuBar mainMenuBar = new JMenuBar();
		mainMenuBar.add(menu);
		return mainMenuBar;
	}

	private void createMenu()
	{
		menu = new JMenu("Options");
		menu.setMnemonic(KeyEvent.VK_O);
		JMenuItem tempMenuItem;
		tempMenuItem = new JMenuItem("Create Agents");
		tempMenuItem.addActionListener(this);
		tempMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, 0));
		menu.add(tempMenuItem);
		menu.addSeparator();
		tempMenuItem = new JMenuItem("Start All Agents");
		tempMenuItem.addActionListener(this);
		tempMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, 0));
		menu.add(tempMenuItem);
		tempMenuItem = new JMenuItem("Stop All Agents");
		tempMenuItem.addActionListener(this);
		tempMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, 0));
		menu.add(tempMenuItem);
		menu.addSeparator();
		tempMenuItem = new JMenuItem("Exit");
		tempMenuItem.addActionListener(this);
		tempMenuItem.setMnemonic(KeyEvent.VK_X);
		menu.add(tempMenuItem);

		menu.addActionListener(this);
	}

	public void step()
	{
		if(agentPanel != null)
			agentPanel.repaint();
	}

	public void actionPerformed(ActionEvent e)
	{
		String label = ((AbstractButton)e.getSource()).getText();

		if(label.equals("Create Agents"))
		{
			String[] messages = { "How many agents would you like?" };
			GetNumberDialog numberDialog = new GetNumberDialog(frame, messages, 2, 100);
			int n = numberDialog.getValue();
			numberDialog.dispose();

			if(n > 0)
			{
				Signal s = new Signal(Signal.GENERATE_AGENTS);
				s.content = new Integer(n);
				master.signal(s);
			}
		}
		else if(label.equals("Start All Agents"))
		{
			Signal s = new Signal(Signal.ALL_STARTED);
			master.signal(s);			
		}
		else if(label.equals("Stop All Agents"))
		{
			Signal s = new Signal(Signal.ALL_STOPPED);
			master.signal(s);			
		}
		else if(label.equals("Exit"))
			System.exit(0);
		else
			System.out.println(label);

		frame.repaint();

	}

	public void rectangleSelected(Rectangle area, int buttonMask)
	{
		switch(buttonMask)
		{
			case SelectionMaker.BUTTON_1:
			{
				Signal s = new Signal(Signal.AREA_STARTED);
				s.content = area;
				master.signal(s);
				break;
			}
			default:
			{
				Signal s = new Signal(Signal.AREA_STOPPED);
				s.content = area;
				master.signal(s);
				break;
			}
		}
	}
	
	public void pointSelected(Point point, int buttonMask)
	{
		switch(buttonMask)
		{
			case SelectionMaker.BUTTON_1:
			{
				Signal s = new Signal(Signal.DESTINATION_CHANGED);
				s.content = point;
				master.signal(s);
				break;
			}
			default:
			{
				Signal s = new Signal(Signal.GENERATE_AGENTS);
				s.content = point;
				master.signal(s);
				break;
			}
		}
	}
}

class SelectionMaker implements MouseListener, MouseMotionListener
{
	public static final int NONE = 0;
	public static final int BUTTON_1 = MouseEvent.BUTTON1_MASK;
	public static final int BUTTON_2 = MouseEvent.BUTTON2_MASK;
	public static final int BUTTON_3 = MouseEvent.BUTTON3_MASK;

	private int selectType = NONE;
	private Point beginPoint;
	private Point endPoint;
	private float alpha = .2f;
	public Color buttonOneColor;
	public Color buttonTwoColor;
	public Color buttonThreeColor;

	SelectionListener listener;

	public SelectionMaker()
	{
		buttonOneColor = Color.green;
		buttonTwoColor = Color.red;
		buttonThreeColor = Color.blue;
	}

	public void addSelectionListener(SelectionListener listener)
	{
		this.listener = listener;
	}

	public void paint(Graphics g)
	{
		if(selectType != NONE)
		{
			Graphics2D g2 = (Graphics2D)g;
			switch(selectType)
			{
				case BUTTON_1: g2.setColor(buttonOneColor); break;
				case BUTTON_2: g2.setColor(buttonTwoColor); break;
				case BUTTON_3: g2.setColor(buttonThreeColor); break;
			}
			g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha));
			g2.fill(makeRectangle(beginPoint, endPoint));
			g2.setComposite(AlphaComposite.Clear);
		}
	}

	public static Rectangle makeRectangle(Point p1, Point p2)
	{
		if(p1 == null && p2 == null)
			return null;
		else if(p1 == null)
			return new Rectangle(p2);
		else if(p2 == null)
			return new Rectangle(p1);
		else
			return new Rectangle(Math.min(p1.x, p2.x), Math.min(p1.y, p2.y),
				Math.abs(p1.x - p2.x), Math.abs(p1.y - p2.y));
	}

	public void mousePressed(MouseEvent e)
	{
		beginPoint = e.getPoint();

		if(SwingUtilities.isMiddleMouseButton(e))
			selectType = BUTTON_3;
		else if(SwingUtilities.isLeftMouseButton(e))
			selectType = BUTTON_2;
		else
			selectType = BUTTON_1;
	}

	public void mouseDragged(MouseEvent e)
	{
		endPoint = e.getPoint();
	}

	public void mouseReleased(MouseEvent e)
	{
		if(listener != null && endPoint != null)
			listener.rectangleSelected(makeRectangle(beginPoint, endPoint), selectType);
		selectType = NONE;
		beginPoint = null;
		endPoint = null;
	}

	public void mouseClicked(MouseEvent e)
	{
		if(SwingUtilities.isMiddleMouseButton(e))
			selectType = BUTTON_3;
		else if(SwingUtilities.isLeftMouseButton(e))
			selectType = BUTTON_2;
		else
			selectType = BUTTON_1;

		if(e.getClickCount() == 2)
			listener.pointSelected(e.getPoint(), selectType);

		selectType = NONE;
	}

	public void mouseMoved(MouseEvent e) {}
	public void mouseEntered(MouseEvent e) {}
	public void mouseExited(MouseEvent e) {}
}

interface SelectionListener
{
	public void rectangleSelected(Rectangle area, int buttonMask);
	public void pointSelected(Point point, int buttonMask);
}

class GetNumberDialog extends JDialog
{
	private int value;
	private JOptionPane optionPane;

	public int getValue()
	{
		return value;
	}

	public GetNumberDialog(Frame owner, String[] messages, final int lowerBound, final int upperBound)
	{
		super(owner, true);
		int textLength = 10;
		setTitle("Number of agents?");
		int i = 0;

		Object[] elements = new Object[messages.length + 1];
		for(i = 0; i < messages.length; i++)
			elements[i] = messages[i];
		final JTextField textField = new JTextField(textLength);
		elements[i] = textField;

		final String btnString1 = "Enter";
		final String btnString2 = "Cancel";
		Object[] options = { btnString1, btnString2 };

		optionPane = new JOptionPane(
		  elements, 
		  JOptionPane.QUESTION_MESSAGE,
		  JOptionPane.YES_NO_OPTION,
		  null,
		  options,
		  options[0]);

		setContentPane(optionPane);
		setLocationRelativeTo(owner);

		pack();

		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
		addWindowListener(new WindowAdapter()
		 {
		  public void windowClosing(WindowEvent we)
		  {
			/* Instead of directly closing the window,
			 * we're going to change the JOptionPane's
			 * value property.
			 */
			 optionPane.setValue(new Integer(JOptionPane.CLOSED_OPTION));
		  }
		 });

		textField.addActionListener(new ActionListener()
		 {
		  public void actionPerformed(ActionEvent e)
		  {
			optionPane.setValue(btnString1);
		  }
		 });

		optionPane.addPropertyChangeListener(new PropertyChangeListener()
		 {
		  public void propertyChange(PropertyChangeEvent e)
		  {
			String prop = e.getPropertyName();

			if(!isVisible()  || e.getSource() != optionPane)
				return;

			if(prop.equals(JOptionPane.VALUE_PROPERTY) ||
			   prop.equals(JOptionPane.INPUT_VALUE_PROPERTY))
			{
				Object optionValue = optionPane.getValue();
				if(optionValue == JOptionPane.UNINITIALIZED_VALUE)
				{
					return;
				}

				/* Reset the JOptionPane's value.
				 * If you don't do this, then if the user
				 * presses the same button next time, no
				 * property change event will be fired.
				 */
				optionPane.setValue(JOptionPane.UNINITIALIZED_VALUE);

				if(optionValue.equals(btnString1))
				{
					try
					{
						value = Integer.parseInt(textField.getText());
					}
					catch(NumberFormatException ne)
					{
						if(textField.getText().length() != 0)
							JOptionPane.showMessageDialog(
							 GetNumberDialog.this,
							 "Sorry, \"" + textField.getText() + "\" "
							 + "isn't a valid number.\n"
							 + "Randomly generating a number between "
							 + lowerBound + " and " + upperBound + ".",
							 "Sure thing",
							 JOptionPane.ERROR_MESSAGE);
						value = lowerBound
						 + (int)(Math.random() * (upperBound - lowerBound));
					}
					setVisible(false);
				}
				else // user closed dialog or clicked cancel
				{
					value = 0;
					setVisible(false);
				}
			}
		  }
		 });
		setVisible(true);
	}
}