import java.awt.*;
import java.awt.geom.*;

public abstract class MovingAgent extends Agent
{
	private static final int MIN_STATE = 1;
	public  static final int MOVING = 1;
	public  static final int BACKTRACKING = 2;
	public  static final int STUCK = 3;
	public  static final int WAITING = 4;
	public  static final int TURNING = 5;
	public  static final int FINISHED = 6;
	public  static final int STOPPED = 7;
	private static final int MAX_STATE = 7;

	public float alpha = .8f;
	public Color[] colors;

	protected Agent master;
	protected int state;
	protected Shape shape;
	protected VirtualWorld world;
	protected AffineTransform rotationTransform = new AffineTransform();
	protected AffineTransform translationTransform = new AffineTransform();
	protected AffineTransform transform = new AffineTransform();
	protected Point position;

	public MovingAgent(Agent master, VirtualWorld world)
	{
		this.master = master;
		this.world = world;

		colors = new Color[MAX_STATE + 1];
		colors[MOVING] = new Color(48, 48, 214);
		colors[BACKTRACKING] = new Color(255, 6, 31);
		colors[STUCK] = new Color(51, 132, 47);
		colors[WAITING] = new Color(142, 143, 185);
		colors[TURNING] = new Color(68, 51, 219);
		colors[FINISHED] = new Color(113, 154, 157);
		colors[STOPPED] = new Color(45, 65, 66);
	}

	public void paint(Graphics g)
	{
		if(shape == null || g == null)
			return;

		Graphics2D g2 = (Graphics2D)g;
		g2.setColor(colors[state]);
		g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha));
		g2.setTransform(transform);
		g2.fill(shape);
		g2.setComposite(AlphaComposite.Clear);
		g2.setTransform(new AffineTransform());
	}

	public Shape getShape()
	{
		Area area = new Area(shape);
		area.transform(transform);
		return area;
	}

	public int getState() { return state; }
	public Agent getMaster() { return master; }

	public void setState(int newState)
	{
		if(newState < MIN_STATE || newState > MAX_STATE)
			return;
		else
			state = newState;
	}

	public synchronized void move(Point deltaPosition)
	{
		transform.translate(deltaPosition.x, deltaPosition.y);
		translationTransform.setToTranslation(transform.getTranslateX(), transform.getTranslateY());
		position.x = 0;
		position.y = 0;
		translationTransform.transform(position, position);
	}

	public synchronized void moveTo(Point position)
	{
		this.position = position;
		translationTransform.setToTranslation(position.x, position.y);
		updateTranslation();
	}

	public synchronized void turn(double theta)
	{
		rotationTransform.rotate(theta);
		updateTranslation();
	}

	public synchronized void turnTo(double theta)
	{
		rotationTransform.setToRotation(theta);
		updateTranslation();
	}

	private synchronized void updateTranslation()
	{
		transform.setToIdentity();
		transform.concatenate(translationTransform);
		transform.concatenate(rotationTransform);
	}

	public boolean intersects(MovingAgent agent)
	{
		if(agent == this)
			return false;
		else
			return intersects(agent.getShape());
	}

	public boolean intersects(Shape shape)
	{
		return ShapeUtilities.isIntersection(getShape(), shape);
	}
}